﻿using System;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;

namespace PWS.API.Input
{
    public static class InputAPI
    {
        /// <summary>
        /// محدود کردن ماوس در یک چهارگوش
        /// </summary>
        /// <param name="lpRect">مختصات چهارگوش</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool ClipCursor(ref Rectangle lpRect);

        /// <summary>
        /// بازیابی محدوده چهار گوش فعالیت ماوس
        /// </summary>
        /// <param name="lpRect">چهار گوش خروجی محدوده ماوس</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool GetClipCursor(out Rectangle lpRect);

        /// <summary>
        ///  مختصات نقطه داده شده در فرم مورد نظر را به مختصات در کل صفحه تبدیل می کند
        /// </summary>
        /// <param name="hWnd">هندل پنجره مورد نظر</param>
        /// <param name="lpPoint">نقطه مورد نظر در پنجره ذکر شده</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool ClientToScreen(
            IntPtr hWnd, ref Point lpPoint);

        /// <summary>
        /// بازیابی اشاره گر جاری ماوس
        /// </summary>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern IntPtr GetCursor();

        /// <summary>
        /// مکان جاری اشاره گر ماوس را مشخص می کند
        /// </summary>
        /// <param name="lpPoint">مکان جاری اشاره گر ماوس</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool GetCursorPos(out Point lpPoint);

        /// <summary>
        /// بردن اشاره گر ماوس به نقطه دلخواه
        /// </summary>
        /// <param name="x">محور افقی</param>
        /// <param name="y">محور عمودی</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool SetCursorPos(int x, int y);

        /// <summary>
        /// نوع  شمارشی از انواع اشره گر های ماوس در ویندوز
        /// </summary>
        public enum IDCStandardCursors
        {
            IDC_Arrow = 0x7F00,
            IDC_Ibeam = 0x7F01,
            IDC_Wait = 0x7F02,
            IDC_Cross = 0x7F03,
            IDC_UpArrow = 0x7F04,
            IDC_Size = 0x7F80,
            IDC_Icon = 0x7F81,
            IDC_SizeNWSE = 0x7F82,
            IDC_SizeNESW = 0x7F83,
            IDC_SizeWE = 0x7F84,
            IDC_SizeNS = 0x7F85,
            IDC_SizeAll = 0x7F86,
            IDC_No = 0x7F88,
            IDC_Hand = 0x7F89,
            IDC_AppStarting = 0x7F8A,
            IDC_Help = 0x7F8B
        }

        /// <summary>
        /// بارگذاری یک کرسر ماوس از یک فایل منبع یا بارگذاری یکی از کرسور های پیش فرض ویندوز
        /// </summary>
        /// <param name="hInstance">آدرس فایل منبع</param>
        /// <param name="lpCursorName">نام یا ایندکس کرسور</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern IntPtr LoadCursor(IntPtr hInstance, int lpCursorName);

        /// <summary>
        /// بارگذاری یک فایل کرسور ماوس
        /// </summary>
        /// <param name="lpFileName">مسیر فایل کرسور</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern IntPtr LoadCursorFromFile(string lpFileName);

        /// <summary>
        /// تغییر کرسور های پیش فرض ویندوز
        /// </summary>
        /// <param name="hCur">اشاره گری از کرسور </param>
        /// <param name="id">ثابت عددی کرسور مورد نظر</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool SetSystemCursor(IntPtr hCur, int id);

        /// <summary>
        /// نشانگر ماوس را مخفی و یا آشکار می کند
        /// </summary>
        /// <param name="bShow">مخفی و آشکار بودن نشانگر ماوس را نشان می دهد</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern int ShowCursor(bool bShow);

        /// <summary>
        /// تنظیم زمان بین دابل کلیک
        /// </summary>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern uint GetDoubleClickTime();

        /// <summary>
        /// تنظیم زمان دابل کلیک ماوس
        /// </summary>
        /// <param name="uInterval">زمان مورد نظر</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool SetDoubleClickTime(uint uInterval);

        /// <summary>
        /// جای کلیک راست و کلیک چپ را تعویض می کند
        /// </summary>
        /// <param name="fSwap">پارامتر تعویش کلیک چپ و راست</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool SwapMouseButton(bool fSwap);

        /// <summary>
        /// ماوس را برای یک پنجره خاص اسیر می کند
        /// </summary>
        /// <param name="hWnd">The h WND.</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern IntPtr SetCapture(IntPtr hWnd);

        /// <summary>
        /// ماوس اسیر شده را ازاد می کند
        /// </summary>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool ReleaseCapture();

        [Flags]
        public enum MouseEventFlags : uint
        {
            LeftDown = 0x00000002,
            LeftUp = 0x00000004,
            MiddleDown = 0x00000020,
            MiddleUp = 0x00000040,
            Move = 0x00000001,
            Absolute = 0x00008000,
            RightDown = 0x00000008,
            RightUp = 0x00000010,
            Wheel = 0x00000800,
            XDown = 0x00000080,
            XUp = 0x00000100
        }

        /// <summary>
        /// شبیه سازی عملیا ماوس
        /// </summary>
        /// <param name="dwFlags">چه عملیاتی می خواهید انجام دهید</param>
        /// <param name="dx">The dx.</param>
        /// <param name="dy">The dy.</param>
        /// <param name="dwData">The dw data.</param>
        /// <param name="dwExtraInfo">The dw extra info.</param>
        [DllImport("user32.dll")]
        public static extern void mouse_event(
            MouseEventFlags dwFlags, 
            int dx, int dy, 
            int dwData, 
            int dwExtraInfo);

        /// <summary>
        /// ماوس و صفحه کلید را قفل و آزاد می کند
        /// </summary>
        /// <param name="fBlockIt">if set to <c>true</c> [f block it].</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool BlockInput(bool fBlockIt);

        /// <summary>
        ///
        /// </summary>
        public enum HKL
        {
            /// <summary>
            /// Selects the previous locale identifier in the circular list of loaded locale identifiers maintained by the system
            /// </summary>
            HKL_Prev = 0,
            /// <summary>
            /// Selects the next locale identifier in the circular list of loaded locale identifiers maintained by the system
            /// </summary>
            HKL_Next = 1
        }

        /// <summary>
        /// این تابع یکی از زبانهای نصب شده در ویندوز را فعال می کند
        /// </summary>
        /// <param name="hkl">یک انتخاب زبان بعدی و صفر انتخاب زبان قبلی</param>
        /// <param name="uFlags">با صفر مقدار دهی کنید</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool ActivateKeyboardLayout(HKL hkl, uint uFlags);

        /// <summary>
        /// این تابع کد عددی زبان جاری صفحه کلید را بر می¬گرداند ؛ در واقع تعیین می کند که صفحه کلید در حالت فارسی است و یا انگلیسی و یا هر زبان دیگری است
        /// </summary>
        /// <param name="pwszKLID">کد زبان جاری را برمی گرداند</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool GetKeyboardLayoutName([Out] StringBuilder pwszKLID);

        /// <summary>
        /// این تابع زبان صفحه کلید را تغیر می دهد
        /// </summary>
        /// <param name="pwszKLID">کد زبان مورد نظر</param>
        /// <param name="uFlags">
        /// 1 - تغییر زبان انجام می شود
        /// 0 - تغییر زبان انجام نمی شود
        /// </param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern IntPtr LoadKeyboardLayout(
            string pwszKLID, uint uFlags);

        /// <summary>
        /// این تابع عدد باینری یک زبان را گرفته و نام ان را بر می گرداند
        /// </summary>
        /// <param name="wLang">کد باینری زبان
        /// 0 - Language Neutral
        /// 1033 - English (United States)
        /// 1040 - Italian (Standard)
        /// ,...
        /// </param>
        /// <param name="szLang">بافر خروجی</param>
        /// <param name="wSize">طول بافر</param>
        /// <returns></returns>
        [DllImport("kernel32.dll")]
        public static extern bool VerLanguageName(
            uint wLang, [Out] StringBuilder szLang, int wSize);

        /// <summary>
        /// نمایش نشانگر ساخته شده با تابع CreateCaret
        /// </summary>
        /// <param name="hWnd">هندل کنترل مقصد</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool ShowCaret(IntPtr hWnd);

        /// <summary>
        /// مخفی کردن نشانگر ساخته شده روی کنترل مورد نظر
        /// </summary>
        /// <param name="hWnd"></param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool HideCaret(IntPtr hWnd);

        /// <summary>
        /// ساخت نشانگر چشمک زن صفحه کلید برای یک کنترل خاص
        /// </summary>
        /// <param name="hWnd">هندل پنجره مورد نظر</param>
        /// <param name="hBitmap">هندل تصویر</param>
        /// <param name="nWidth">عرض نشانگر</param>
        /// <param name="nHeight">ارتفاع نشانگر</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool CreateCaret(
            IntPtr hWnd, IntPtr hBitmap, int nWidth, int nHeight);

        /// <summary>
        /// بازیابی زمان چشمک زدن نشانگر چشمک زن صفحه کلید
        /// </summary>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern uint GetCaretBlinkTime();

        /// <summary>
        /// تعیین زمان چشمک زدن نشانگر چشمک زن صفحه کلید
        /// </summary>
        /// <param name="uMSeconds">زمان به میلی ثانیه</param>
        /// <returns></returns>
        [DllImport("user32.dll")]
        public static extern bool SetCaretBlinkTime(uint uMSeconds);

        [DllImport("user32.dll")]
        public static extern bool GetKeyboardState(byte[] lpKeyState);

        [DllImport("user32.dll")]
        public static extern bool SetKeyboardState(byte[] lpKeyState);

        [DllImport("user32.dll")]
        public static extern int GetKeyboardType(int nTypeFlag);

        [DllImport("user32.dll")]
        public static extern short GetKeyState(VirtualKeyStates nVirtKey);

        public enum VirtualKeyStates : int
        {
            VK_LBUTTON = 0x01,
            VK_RBUTTON = 0x02,
            VK_CANCEL = 0x03,
            VK_MBUTTON = 0x04,
            //
            VK_XBUTTON1 = 0x05,
            VK_XBUTTON2 = 0x06,
            //
            VK_BACK = 0x08,
            VK_TAB = 0x09,
            //
            VK_CLEAR = 0x0C,
            VK_RETURN = 0x0D,
            //
            VK_SHIFT = 0x10,
            VK_CONTROL = 0x11,
            VK_MENU = 0x12,
            VK_PAUSE = 0x13,
            VK_CAPITAL = 0x14,
            //
            VK_KANA = 0x15,
            VK_HANGEUL = 0x15,  /* old name - should be here for compatibility */
            VK_HANGUL = 0x15,
            VK_JUNJA = 0x17,
            VK_FINAL = 0x18,
            VK_HANJA = 0x19,
            VK_KANJI = 0x19,
            //
            VK_ESCAPE = 0x1B,
            //
            VK_CONVERT = 0x1C,
            VK_NONCONVERT = 0x1D,
            VK_ACCEPT = 0x1E,
            VK_MODECHANGE = 0x1F,
            //
            VK_SPACE = 0x20,
            VK_PRIOR = 0x21,
            VK_NEXT = 0x22,
            VK_END = 0x23,
            VK_HOME = 0x24,
            VK_LEFT = 0x25,
            VK_UP = 0x26,
            VK_RIGHT = 0x27,
            VK_DOWN = 0x28,
            VK_SELECT = 0x29,
            VK_PRINT = 0x2A,
            VK_EXECUTE = 0x2B,
            VK_SNAPSHOT = 0x2C,
            VK_INSERT = 0x2D,
            VK_DELETE = 0x2E,
            VK_HELP = 0x2F,
            //
            VK_LWIN = 0x5B,
            VK_RWIN = 0x5C,
            VK_APPS = 0x5D,
            //
            VK_SLEEP = 0x5F,
            //
            VK_NUMPAD0 = 0x60,
            VK_NUMPAD1 = 0x61,
            VK_NUMPAD2 = 0x62,
            VK_NUMPAD3 = 0x63,
            VK_NUMPAD4 = 0x64,
            VK_NUMPAD5 = 0x65,
            VK_NUMPAD6 = 0x66,
            VK_NUMPAD7 = 0x67,
            VK_NUMPAD8 = 0x68,
            VK_NUMPAD9 = 0x69,
            VK_MULTIPLY = 0x6A,
            VK_ADD = 0x6B,
            VK_SEPARATOR = 0x6C,
            VK_SUBTRACT = 0x6D,
            VK_DECIMAL = 0x6E,
            VK_DIVIDE = 0x6F,
            VK_F1 = 0x70,
            VK_F2 = 0x71,
            VK_F3 = 0x72,
            VK_F4 = 0x73,
            VK_F5 = 0x74,
            VK_F6 = 0x75,
            VK_F7 = 0x76,
            VK_F8 = 0x77,
            VK_F9 = 0x78,
            VK_F10 = 0x79,
            VK_F11 = 0x7A,
            VK_F12 = 0x7B,
            VK_F13 = 0x7C,
            VK_F14 = 0x7D,
            VK_F15 = 0x7E,
            VK_F16 = 0x7F,
            VK_F17 = 0x80,
            VK_F18 = 0x81,
            VK_F19 = 0x82,
            VK_F20 = 0x83,
            VK_F21 = 0x84,
            VK_F22 = 0x85,
            VK_F23 = 0x86,
            VK_F24 = 0x87,
            //
            VK_NUMLOCK = 0x90,
            VK_SCROLL = 0x91,
            //
            VK_OEM_NEC_EQUAL = 0x92,   // '=' key on numpad
            //
            VK_OEM_FJ_JISHO = 0x92,   // 'Dictionary' key
            VK_OEM_FJ_MASSHOU = 0x93,   // 'Unregister word' key
            VK_OEM_FJ_TOUROKU = 0x94,   // 'Register word' key
            VK_OEM_FJ_LOYA = 0x95,   // 'Left OYAYUBI' key
            VK_OEM_FJ_ROYA = 0x96,   // 'Right OYAYUBI' key
            //
            VK_LSHIFT = 0xA0,
            VK_RSHIFT = 0xA1,
            VK_LCONTROL = 0xA2,
            VK_RCONTROL = 0xA3,
            VK_LMENU = 0xA4,
            VK_RMENU = 0xA5,
            //
            VK_BROWSER_BACK = 0xA6,
            VK_BROWSER_FORWARD = 0xA7,
            VK_BROWSER_REFRESH = 0xA8,
            VK_BROWSER_STOP = 0xA9,
            VK_BROWSER_SEARCH = 0xAA,
            VK_BROWSER_FAVORITES = 0xAB,
            VK_BROWSER_HOME = 0xAC,
            //
            VK_VOLUME_MUTE = 0xAD,
            VK_VOLUME_DOWN = 0xAE,
            VK_VOLUME_UP = 0xAF,
            VK_MEDIA_NEXT_TRACK = 0xB0,
            VK_MEDIA_PREV_TRACK = 0xB1,
            VK_MEDIA_STOP = 0xB2,
            VK_MEDIA_PLAY_PAUSE = 0xB3,
            VK_LAUNCH_MAIL = 0xB4,
            VK_LAUNCH_MEDIA_SELECT = 0xB5,
            VK_LAUNCH_APP1 = 0xB6,
            VK_LAUNCH_APP2 = 0xB7,
            //
            VK_OEM_1 = 0xBA,   // ';:' for US
            VK_OEM_PLUS = 0xBB,   // '+' any country
            VK_OEM_COMMA = 0xBC,   // ',' any country
            VK_OEM_MINUS = 0xBD,   // '-' any country
            VK_OEM_PERIOD = 0xBE,   // '.' any country
            VK_OEM_2 = 0xBF,   // '/?' for US
            VK_OEM_3 = 0xC0,   // '`~' for US
            //
            VK_OEM_4 = 0xDB,  //  '[{' for US
            VK_OEM_5 = 0xDC,  //  '\|' for US
            VK_OEM_6 = 0xDD,  //  ']}' for US
            VK_OEM_7 = 0xDE,  //  ''"' for US
            VK_OEM_8 = 0xDF,
            //
            VK_OEM_AX = 0xE1,  //  'AX' key on Japanese AX kbd
            VK_OEM_102 = 0xE2,  //  "<>" or "\|" on RT 102-key kbd.
            VK_ICO_HELP = 0xE3,  //  Help key on ICO
            VK_ICO_00 = 0xE4,  //  00 key on ICO
            //
            VK_PROCESSKEY = 0xE5,
            //
            VK_ICO_CLEAR = 0xE6,
            //
            VK_PACKET = 0xE7,
            //
            VK_OEM_RESET = 0xE9,
            VK_OEM_JUMP = 0xEA,
            VK_OEM_PA1 = 0xEB,
            VK_OEM_PA2 = 0xEC,
            VK_OEM_PA3 = 0xED,
            VK_OEM_WSCTRL = 0xEE,
            VK_OEM_CUSEL = 0xEF,
            VK_OEM_ATTN = 0xF0,
            VK_OEM_FINISH = 0xF1,
            VK_OEM_COPY = 0xF2,
            VK_OEM_AUTO = 0xF3,
            VK_OEM_ENLW = 0xF4,
            VK_OEM_BACKTAB = 0xF5,
            //
            VK_ATTN = 0xF6,
            VK_CRSEL = 0xF7,
            VK_EXSEL = 0xF8,
            VK_EREOF = 0xF9,
            VK_PLAY = 0xFA,
            VK_ZOOM = 0xFB,
            VK_NONAME = 0xFC,
            VK_PA1 = 0xFD,
            VK_OEM_CLEAR = 0xFE
        }

        [DllImport("user32.dll")]
        public static extern short GetAsyncKeyState(Keys vKey);

        public const uint KeyEvent_ExtendedKey = 0x01;
        public const uint KeyEvent_KeyUp = 0x02;
        public const uint KeyEvent_KeyDown = 0x0;

        [DllImport("user32.dll")]
        public static extern void keybd_event(
            Keys bVk, byte bScan, uint dwFlags, UIntPtr dwExtraInfo);

        /// <summary>
        /// Synthesizes keystrokes, mouse motions, and button clicks.
        /// </summary>
        [DllImport("user32.dll")]
        public static extern UInt32 SendInput(
            UInt32 numberOfInputs, Input[] inputs, Int32 sizeOfInputStructure);

        public struct Input
        {
            public UInt32 Type;

            public MouseKeyboardHardwareInput Data;

            public static Int32 SizeOf
            {
                get { return Marshal.SizeOf(typeof(Input)); }
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        public struct MouseKeyboardHardwareInput
        {
            [FieldOffset(0)]
            public MouseInput Mouse;

            [FieldOffset(0)]
            public KeyboardInput Keyboard;

            [FieldOffset(0)]
            public HardwareInput Hardware;
        }

        public enum InputType : uint // UInt32
        {
            Mouse = 0,
            Keyboard = 1,
            Hardware = 2,
        }

        public struct MouseInput
        {
            public Int32 X;
            public Int32 Y;
            public UInt32 MouseData;
            public UInt32 Flags;
            public UInt32 Time;
            public IntPtr ExtraInfo;
        }

        public struct KeyboardInput
        {
            public UInt16 KeyCode;
            public UInt16 Scan;
            public UInt32 Flags;
            public UInt32 Time;
            public IntPtr ExtraInfo;
        }

        public struct HardwareInput
        {
            public UInt32 Msg;
            public UInt16 ParamL;
            public UInt16 ParamH;
        }

        [Flags]
        public enum KeyboardFlag : uint // UInt32
        {
            ExtendedKey = 0x0001,
            KeyUp = 0x0002,
            Unicode = 0x0004,
            ScanCode = 0x0008,
        }

        public enum VirtualKeyCode : ushort // UInt16
        {
            LBUTTON = 0x01,

            RBUTTON = 0x02,

            CANCEL = 0x03,

            /// <summary>
            /// Middle mouse button (three-button mouse) - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            MBUTTON = 0x04,

            /// <summary>
            /// Windows 2000/XP: X1 mouse button - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            XBUTTON1 = 0x05,

            /// <summary>
            /// Windows 2000/XP: X2 mouse button - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            XBUTTON2 = 0x06,

            // 0x07 : Undefined

            /// <summary>
            /// BACKSPACE key
            /// </summary>
            BACK = 0x08,

            /// <summary>
            /// TAB key
            /// </summary>
            TAB = 0x09,

            // 0x0A - 0x0B : Reserved

            /// <summary>
            /// CLEAR key
            /// </summary>
            CLEAR = 0x0C,

            /// <summary>
            /// ENTER key
            /// </summary>
            RETURN = 0x0D,

            // 0x0E - 0x0F : Undefined

            /// <summary>
            /// SHIFT key
            /// </summary>
            SHIFT = 0x10,

            /// <summary>
            /// CTRL key
            /// </summary>
            CONTROL = 0x11,

            /// <summary>
            /// ALT key
            /// </summary>
            MENU = 0x12,

            /// <summary>
            /// PAUSE key
            /// </summary>
            PAUSE = 0x13,

            /// <summary>
            /// CAPS LOCK key
            /// </summary>
            CAPITAL = 0x14,

            /// <summary>
            /// Input Method Editor (IME) Kana mode
            /// </summary>
            KANA = 0x15,

            /// <summary>
            /// IME Hanguel mode (maintained for compatibility; use HANGUL)
            /// </summary>
            HANGEUL = 0x15,

            /// <summary>
            /// IME Hangul mode
            /// </summary>
            HANGUL = 0x15,

            // 0x16 : Undefined

            /// <summary>
            /// IME Junja mode
            /// </summary>
            JUNJA = 0x17,

            /// <summary>
            /// IME final mode
            /// </summary>
            FINAL = 0x18,

            /// <summary>
            /// IME Hanja mode
            /// </summary>
            HANJA = 0x19,

            /// <summary>
            /// IME Kanji mode
            /// </summary>
            KANJI = 0x19,

            // 0x1A : Undefined

            /// <summary>
            /// ESC key
            /// </summary>
            ESCAPE = 0x1B,

            /// <summary>
            /// IME convert
            /// </summary>
            CONVERT = 0x1C,

            /// <summary>
            /// IME nonconvert
            /// </summary>
            NONCONVERT = 0x1D,

            /// <summary>
            /// IME accept
            /// </summary>
            ACCEPT = 0x1E,

            /// <summary>
            /// IME mode change request
            /// </summary>
            MODECHANGE = 0x1F,

            /// <summary>
            /// SPACEBAR
            /// </summary>
            SPACE = 0x20,

            /// <summary>
            /// PAGE UP key
            /// </summary>
            PRIOR = 0x21,

            /// <summary>
            /// PAGE DOWN key
            /// </summary>
            NEXT = 0x22,

            /// <summary>
            /// END key
            /// </summary>
            END = 0x23,

            /// <summary>
            /// HOME key
            /// </summary>
            HOME = 0x24,

            /// <summary>
            /// LEFT ARROW key
            /// </summary>
            LEFT = 0x25,

            /// <summary>
            /// UP ARROW key
            /// </summary>
            UP = 0x26,

            /// <summary>
            /// RIGHT ARROW key
            /// </summary>
            RIGHT = 0x27,

            /// <summary>
            /// DOWN ARROW key
            /// </summary>
            DOWN = 0x28,

            /// <summary>
            /// SELECT key
            /// </summary>
            SELECT = 0x29,

            /// <summary>
            /// PRINT key
            /// </summary>
            PRINT = 0x2A,

            /// <summary>
            /// EXECUTE key
            /// </summary>
            EXECUTE = 0x2B,

            /// <summary>
            /// PRINT SCREEN key
            /// </summary>
            SNAPSHOT = 0x2C,

            /// <summary>
            /// INS key
            /// </summary>
            INSERT = 0x2D,

            /// <summary>
            /// DEL key
            /// </summary>
            DELETE = 0x2E,

            /// <summary>
            /// HELP key
            /// </summary>
            HELP = 0x2F,

            /// <summary>
            /// 0 key
            /// </summary>
            VK_0 = 0x30,

            /// <summary>
            /// 1 key
            /// </summary>
            VK_1 = 0x31,

            /// <summary>
            /// 2 key
            /// </summary>
            VK_2 = 0x32,

            /// <summary>
            /// 3 key
            /// </summary>
            VK_3 = 0x33,

            /// <summary>
            /// 4 key
            /// </summary>
            VK_4 = 0x34,

            /// <summary>
            /// 5 key
            /// </summary>
            VK_5 = 0x35,

            /// <summary>
            /// 6 key
            /// </summary>
            VK_6 = 0x36,

            /// <summary>
            /// 7 key
            /// </summary>
            VK_7 = 0x37,

            /// <summary>
            /// 8 key
            /// </summary>
            VK_8 = 0x38,

            /// <summary>
            /// 9 key
            /// </summary>
            VK_9 = 0x39,

            //
            // 0x3A - 0x40 : Udefined
            //

            /// <summary>
            /// A key
            /// </summary>
            VK_A = 0x41,

            /// <summary>
            /// B key
            /// </summary>
            VK_B = 0x42,

            /// <summary>
            /// C key
            /// </summary>
            VK_C = 0x43,

            /// <summary>
            /// D key
            /// </summary>
            VK_D = 0x44,

            /// <summary>
            /// E key
            /// </summary>
            VK_E = 0x45,

            /// <summary>
            /// F key
            /// </summary>
            VK_F = 0x46,

            /// <summary>
            /// G key
            /// </summary>
            VK_G = 0x47,

            /// <summary>
            /// H key
            /// </summary>
            VK_H = 0x48,

            /// <summary>
            /// I key
            /// </summary>
            VK_I = 0x49,

            /// <summary>
            /// J key
            /// </summary>
            VK_J = 0x4A,

            /// <summary>
            /// K key
            /// </summary>
            VK_K = 0x4B,

            /// <summary>
            /// L key
            /// </summary>
            VK_L = 0x4C,

            /// <summary>
            /// M key
            /// </summary>
            VK_M = 0x4D,

            /// <summary>
            /// N key
            /// </summary>
            VK_N = 0x4E,

            /// <summary>
            /// O key
            /// </summary>
            VK_O = 0x4F,

            /// <summary>
            /// P key
            /// </summary>
            VK_P = 0x50,

            /// <summary>
            /// Q key
            /// </summary>
            VK_Q = 0x51,

            /// <summary>
            /// R key
            /// </summary>
            VK_R = 0x52,

            /// <summary>
            /// S key
            /// </summary>
            VK_S = 0x53,

            /// <summary>
            /// T key
            /// </summary>
            VK_T = 0x54,

            /// <summary>
            /// U key
            /// </summary>
            VK_U = 0x55,

            /// <summary>
            /// V key
            /// </summary>
            VK_V = 0x56,

            /// <summary>
            /// W key
            /// </summary>
            VK_W = 0x57,

            /// <summary>
            /// X key
            /// </summary>
            VK_X = 0x58,

            /// <summary>
            /// Y key
            /// </summary>
            VK_Y = 0x59,

            /// <summary>
            /// Z key
            /// </summary>
            VK_Z = 0x5A,

            /// <summary>
            /// Left Windows key (Microsoft Natural keyboard)
            /// </summary>
            LWIN = 0x5B,

            /// <summary>
            /// Right Windows key (Natural keyboard)
            /// </summary>
            RWIN = 0x5C,

            /// <summary>
            /// Applications key (Natural keyboard)
            /// </summary>
            APPS = 0x5D,

            // 0x5E : reserved

            /// <summary>
            /// Computer Sleep key
            /// </summary>
            SLEEP = 0x5F,

            /// <summary>
            /// Numeric keypad 0 key
            /// </summary>
            NUMPAD0 = 0x60,

            /// <summary>
            /// Numeric keypad 1 key
            /// </summary>
            NUMPAD1 = 0x61,

            /// <summary>
            /// Numeric keypad 2 key
            /// </summary>
            NUMPAD2 = 0x62,

            /// <summary>
            /// Numeric keypad 3 key
            /// </summary>
            NUMPAD3 = 0x63,

            /// <summary>
            /// Numeric keypad 4 key
            /// </summary>
            NUMPAD4 = 0x64,

            /// <summary>
            /// Numeric keypad 5 key
            /// </summary>
            NUMPAD5 = 0x65,

            /// <summary>
            /// Numeric keypad 6 key
            /// </summary>
            NUMPAD6 = 0x66,

            /// <summary>
            /// Numeric keypad 7 key
            /// </summary>
            NUMPAD7 = 0x67,

            /// <summary>
            /// Numeric keypad 8 key
            /// </summary>
            NUMPAD8 = 0x68,

            /// <summary>
            /// Numeric keypad 9 key
            /// </summary>
            NUMPAD9 = 0x69,

            /// <summary>
            /// Multiply key
            /// </summary>
            MULTIPLY = 0x6A,

            /// <summary>
            /// Add key
            /// </summary>
            ADD = 0x6B,

            /// <summary>
            /// Separator key
            /// </summary>
            SEPARATOR = 0x6C,

            /// <summary>
            /// Subtract key
            /// </summary>
            SUBTRACT = 0x6D,

            /// <summary>
            /// Decimal key
            /// </summary>
            DECIMAL = 0x6E,

            /// <summary>
            /// Divide key
            /// </summary>
            DIVIDE = 0x6F,

            /// <summary>
            /// F1 key
            /// </summary>
            F1 = 0x70,

            /// <summary>
            /// F2 key
            /// </summary>
            F2 = 0x71,

            /// <summary>
            /// F3 key
            /// </summary>
            F3 = 0x72,

            /// <summary>
            /// F4 key
            /// </summary>
            F4 = 0x73,

            /// <summary>
            /// F5 key
            /// </summary>
            F5 = 0x74,

            /// <summary>
            /// F6 key
            /// </summary>
            F6 = 0x75,

            /// <summary>
            /// F7 key
            /// </summary>
            F7 = 0x76,

            /// <summary>
            /// F8 key
            /// </summary>
            F8 = 0x77,

            /// <summary>
            /// F9 key
            /// </summary>
            F9 = 0x78,

            /// <summary>
            /// F10 key
            /// </summary>
            F10 = 0x79,

            /// <summary>
            /// F11 key
            /// </summary>
            F11 = 0x7A,

            /// <summary>
            /// F12 key
            /// </summary>
            F12 = 0x7B,

            /// <summary>
            /// F13 key
            /// </summary>
            F13 = 0x7C,

            /// <summary>
            /// F14 key
            /// </summary>
            F14 = 0x7D,

            /// <summary>
            /// F15 key
            /// </summary>
            F15 = 0x7E,

            /// <summary>
            /// F16 key
            /// </summary>
            F16 = 0x7F,

            /// <summary>
            /// F17 key
            /// </summary>
            F17 = 0x80,

            /// <summary>
            /// F18 key
            /// </summary>
            F18 = 0x81,

            /// <summary>
            /// F19 key
            /// </summary>
            F19 = 0x82,

            /// <summary>
            /// F20 key
            /// </summary>
            F20 = 0x83,

            /// <summary>
            /// F21 key
            /// </summary>
            F21 = 0x84,

            /// <summary>
            /// F22 key
            /// </summary>
            F22 = 0x85,

            /// <summary>
            /// F23 key
            /// </summary>
            F23 = 0x86,

            /// <summary>
            /// F24 key
            /// </summary>
            F24 = 0x87,

            //
            // 0x88 - 0x8F : Unassigned
            //

            /// <summary>
            /// NUM LOCK key
            /// </summary>
            NUMLOCK = 0x90,

            /// <summary>
            /// SCROLL LOCK key
            /// </summary>
            SCROLL = 0x91,

            // 0x92 - 0x96 : OEM Specific

            // 0x97 - 0x9F : Unassigned

            //
            // L* & R* - left and right Alt, Ctrl and Shift virtual keys.
            // Used only as parameters to GetAsyncKeyState() and GetKeyState().
            // No other API or message will distinguish left and right keys in this way.
            //

            /// <summary>
            /// Left SHIFT key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LSHIFT = 0xA0,

            /// <summary>
            /// Right SHIFT key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RSHIFT = 0xA1,

            /// <summary>
            /// Left CONTROL key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LCONTROL = 0xA2,

            /// <summary>
            /// Right CONTROL key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RCONTROL = 0xA3,

            /// <summary>
            /// Left MENU key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LMENU = 0xA4,

            /// <summary>
            /// Right MENU key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RMENU = 0xA5,

            /// <summary>
            /// Windows 2000/XP: Browser Back key
            /// </summary>
            BROWSER_BACK = 0xA6,

            /// <summary>
            /// Windows 2000/XP: Browser Forward key
            /// </summary>
            BROWSER_FORWARD = 0xA7,

            /// <summary>
            /// Windows 2000/XP: Browser Refresh key
            /// </summary>
            BROWSER_REFRESH = 0xA8,

            /// <summary>
            /// Windows 2000/XP: Browser Stop key
            /// </summary>
            BROWSER_STOP = 0xA9,

            /// <summary>
            /// Windows 2000/XP: Browser Search key
            /// </summary>
            BROWSER_SEARCH = 0xAA,

            /// <summary>
            /// Windows 2000/XP: Browser Favorites key
            /// </summary>
            BROWSER_FAVORITES = 0xAB,

            /// <summary>
            /// Windows 2000/XP: Browser Start and Home key
            /// </summary>
            BROWSER_HOME = 0xAC,

            /// <summary>
            /// Windows 2000/XP: Volume Mute key
            /// </summary>
            VOLUME_MUTE = 0xAD,

            /// <summary>
            /// Windows 2000/XP: Volume Down key
            /// </summary>
            VOLUME_DOWN = 0xAE,

            /// <summary>
            /// Windows 2000/XP: Volume Up key
            /// </summary>
            VOLUME_UP = 0xAF,

            /// <summary>
            /// Windows 2000/XP: Next Track key
            /// </summary>
            MEDIA_NEXT_TRACK = 0xB0,

            /// <summary>
            /// Windows 2000/XP: Previous Track key
            /// </summary>
            MEDIA_PREV_TRACK = 0xB1,

            /// <summary>
            /// Windows 2000/XP: Stop Media key
            /// </summary>
            MEDIA_STOP = 0xB2,

            /// <summary>
            /// Windows 2000/XP: Play/Pause Media key
            /// </summary>
            MEDIA_PLAY_PAUSE = 0xB3,

            /// <summary>
            /// Windows 2000/XP: Start Mail key
            /// </summary>
            LAUNCH_MAIL = 0xB4,

            /// <summary>
            /// Windows 2000/XP: Select Media key
            /// </summary>
            LAUNCH_MEDIA_SELECT = 0xB5,

            /// <summary>
            /// Windows 2000/XP: Start Application 1 key
            /// </summary>
            LAUNCH_APP1 = 0xB6,

            /// <summary>
            /// Windows 2000/XP: Start Application 2 key
            /// </summary>
            LAUNCH_APP2 = 0xB7,

            //
            // 0xB8 - 0xB9 : Reserved
            //

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the ';:' key
            /// </summary>
            OEM_1 = 0xBA,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '+' key
            /// </summary>
            OEM_PLUS = 0xBB,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the ',' key
            /// </summary>
            OEM_COMMA = 0xBC,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '-' key
            /// </summary>
            OEM_MINUS = 0xBD,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '.' key
            /// </summary>
            OEM_PERIOD = 0xBE,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '/?' key
            /// </summary>
            OEM_2 = 0xBF,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '`~' key
            /// </summary>
            OEM_3 = 0xC0,

            //
            // 0xC1 - 0xD7 : Reserved
            //

            //
            // 0xD8 - 0xDA : Unassigned
            //

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '[{' key
            /// </summary>
            OEM_4 = 0xDB,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '\|' key
            /// </summary>
            OEM_5 = 0xDC,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the ']}' key
            /// </summary>
            OEM_6 = 0xDD,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the 'single-quote/double-quote' key
            /// </summary>
            OEM_7 = 0xDE,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard.
            /// </summary>
            OEM_8 = 0xDF,

            //
            // 0xE0 : Reserved
            //

            //
            // 0xE1 : OEM Specific
            //

            /// <summary>
            /// Windows 2000/XP: Either the angle bracket key or the backslash key on the RT 102-key keyboard
            /// </summary>
            OEM_102 = 0xE2,

            //
            // (0xE3-E4) : OEM specific
            //

            /// <summary>
            /// Windows 95/98/Me, Windows NT 4.0, Windows 2000/XP: IME PROCESS key
            /// </summary>
            PROCESSKEY = 0xE5,

            //
            // 0xE6 : OEM specific
            //

            /// <summary>
            /// Windows 2000/XP: Used to pass Unicode characters as if they were keystrokes. The PACKET key is the low word of a 32-bit Virtual Key value used for non-keyboard input methods. For more information, see Remark in KEYBDINPUT, SendInput, WM_KEYDOWN, and WM_KEYUP
            /// </summary>
            PACKET = 0xE7,

            //
            // 0xE8 : Unassigned
            //

            //
            // 0xE9-F5 : OEM specific
            //

            /// <summary>
            /// Attn key
            /// </summary>
            ATTN = 0xF6,

            /// <summary>
            /// CrSel key
            /// </summary>
            CRSEL = 0xF7,

            /// <summary>
            /// ExSel key
            /// </summary>
            EXSEL = 0xF8,

            /// <summary>
            /// Erase EOF key
            /// </summary>
            EREOF = 0xF9,

            /// <summary>
            /// Play key
            /// </summary>
            PLAY = 0xFA,

            /// <summary>
            /// Zoom key
            /// </summary>
            ZOOM = 0xFB,

            /// <summary>
            /// Reserved
            /// </summary>
            NONAME = 0xFC,

            /// <summary>
            /// PA1 key
            /// </summary>
            PA1 = 0xFD,

            /// <summary>
            /// Clear key
            /// </summary>
            OEM_CLEAR = 0xFE,
        }

        public enum ScanCodeShort : short
        {
            LBUTTON = 0,
            RBUTTON = 0,
            CANCEL = 70,
            MBUTTON = 0,
            XBUTTON1 = 0,
            XBUTTON2 = 0,
            BACK = 14,
            TAB = 15,
            CLEAR = 76,
            RETURN = 28,
            SHIFT = 42,
            CONTROL = 29,
            MENU = 56,
            PAUSE = 0,
            CAPITAL = 58,
            KANA = 0,
            HANGUL = 0,
            JUNJA = 0,
            FINAL = 0,
            HANJA = 0,
            KANJI = 0,
            ESCAPE = 1,
            CONVERT = 0,
            NONCONVERT = 0,
            ACCEPT = 0,
            MODECHANGE = 0,
            SPACE = 57,
            PRIOR = 73,
            NEXT = 81,
            END = 79,
            HOME = 71,
            LEFT = 75,
            UP = 72,
            RIGHT = 77,
            DOWN = 80,
            SELECT = 0,
            PRINT = 0,
            EXECUTE = 0,
            SNAPSHOT = 84,
            INSERT = 82,
            DELETE = 83,
            HELP = 99,
            KEY_0 = 11,
            KEY_1 = 2,
            KEY_2 = 3,
            KEY_3 = 4,
            KEY_4 = 5,
            KEY_5 = 6,
            KEY_6 = 7,
            KEY_7 = 8,
            KEY_8 = 9,
            KEY_9 = 10,
            KEY_A = 30,
            KEY_B = 48,
            KEY_C = 46,
            KEY_D = 32,
            KEY_E = 18,
            KEY_F = 33,
            KEY_G = 34,
            KEY_H = 35,
            KEY_I = 23,
            KEY_J = 36,
            KEY_K = 37,
            KEY_L = 38,
            KEY_M = 50,
            KEY_N = 49,
            KEY_O = 24,
            KEY_P = 25,
            KEY_Q = 16,
            KEY_R = 19,
            KEY_S = 31,
            KEY_T = 20,
            KEY_U = 22,
            KEY_V = 47,
            KEY_W = 17,
            KEY_X = 45,
            KEY_Y = 21,
            KEY_Z = 44,
            LWIN = 91,
            RWIN = 92,
            APPS = 93,
            SLEEP = 95,
            NUMPAD0 = 82,
            NUMPAD1 = 79,
            NUMPAD2 = 80,
            NUMPAD3 = 81,
            NUMPAD4 = 75,
            NUMPAD5 = 76,
            NUMPAD6 = 77,
            NUMPAD7 = 71,
            NUMPAD8 = 72,
            NUMPAD9 = 73,
            MULTIPLY = 55,
            ADD = 78,
            SEPARATOR = 0,
            SUBTRACT = 74,
            DECIMAL = 83,
            DIVIDE = 53,
            F1 = 59,
            F2 = 60,
            F3 = 61,
            F4 = 62,
            F5 = 63,
            F6 = 64,
            F7 = 65,
            F8 = 66,
            F9 = 67,
            F10 = 68,
            F11 = 87,
            F12 = 88,
            F13 = 100,
            F14 = 101,
            F15 = 102,
            F16 = 103,
            F17 = 104,
            F18 = 105,
            F19 = 106,
            F20 = 107,
            F21 = 108,
            F22 = 109,
            F23 = 110,
            F24 = 118,
            NUMLOCK = 69,
            SCROLL = 70,
            LSHIFT = 42,
            RSHIFT = 54,
            LCONTROL = 29,
            RCONTROL = 29,
            LMENU = 56,
            RMENU = 56,
            BROWSER_BACK = 106,
            BROWSER_FORWARD = 105,
            BROWSER_REFRESH = 103,
            BROWSER_STOP = 104,
            BROWSER_SEARCH = 101,
            BROWSER_FAVORITES = 102,
            BROWSER_HOME = 50,
            VOLUME_MUTE = 32,
            VOLUME_DOWN = 46,
            VOLUME_UP = 48,
            MEDIA_NEXT_TRACK = 25,
            MEDIA_PREV_TRACK = 16,
            MEDIA_STOP = 36,
            MEDIA_PLAY_PAUSE = 34,
            LAUNCH_MAIL = 108,
            LAUNCH_MEDIA_SELECT = 109,
            LAUNCH_APP1 = 107,
            LAUNCH_APP2 = 33,
            OEM_1 = 39,
            OEM_PLUS = 13,
            OEM_COMMA = 51,
            OEM_MINUS = 12,
            OEM_PERIOD = 52,
            OEM_2 = 53,
            OEM_3 = 41,
            OEM_4 = 26,
            OEM_5 = 43,
            OEM_6 = 27,
            OEM_7 = 40,
            OEM_8 = 0,
            OEM_102 = 86,
            PROCESSKEY = 0,
            PACKET = 0,
            ATTN = 0,
            CRSEL = 0,
            EXSEL = 0,
            EREOF = 93,
            PLAY = 0,
            ZOOM = 98,
            NONAME = 0,
            PA1 = 0,
            OEM_CLEAR = 0,
        }

        [Flags]
        public enum MouseEventDataXButtons : uint
        {
            Nothing = 0x00000000,
            XButton1 = 0x00000001,
            XButton2 = 0x00000002
        }

        [DllImport("user32.dll")]
        public static extern bool RegisterHotKey(IntPtr hWnd, 
            int id, ModifierFlages fsModifiers, Keys vk);

        [DllImport("user32.dll")]
        public static extern bool UnregisterHotKey(IntPtr hWnd, int id);

        public enum ModifierFlages
        {
            Alt = 0x1,
            Control = 0x2,
            Shift = 0x4,
            Windows = 0x8,
            Left = 0x8000,
            Right = 0x4000
        }
    }
}
